#region Using Statements
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net.Mime;
using System.Security.Cryptography.Xml;
using System.Web.Mvc;
using System.Web.Routing;
using HIPS.CommonSchemas;
using HIPS.PcehrSchemas;
using HIPS.Web.Components.Cache;
using HIPS.Web.Components.Collections;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Components.Web;
using HIPS.Web.Components.Common;
using HIPS.Web.Data.Hips;
using HIPS.Web.Data.Hips.PcehrView;
using HIPS.Web.Data.Hips.Reference;
using HIPS.Web.Data.WebsiteDb;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.PcehrView;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.Filters;

using HIPS.Web.UI.ViewModels.Shared;
using HIPS.Web.UI.ViewModels.PcehrView;
#endregion

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// Controller for the "PCEHR Web Viewer" feature.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
    /// </history>
    [NoCache]
    public class PcehrViewController : Controller
    {

        #region Fields

        /// <summary>
        /// List of application-specific settings.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        private readonly List<Setting> Settings;

        /// <summary>
        /// Gets the hospital repository to be used by this controller.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        private readonly IHospitalRepository HospitalRepository;

        /// <summary>
        /// Gets the patient repository to be used by this controller.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        private readonly IPatientRepository PatientRepository;

        /// <summary>
        /// Gets the PCEHR view repository to be used by this controller.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        private readonly IPcehrViewRepository PcehrViewRepository;

        #endregion

        #region Properties

        /// <summary>
        /// Gets the default hospital code system used by the application.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        private string DefaultHospitalCodeSystem
        {
            get
            {
                return this.Settings.GetSettingValue(Setting.SettingCodes.DefaultHospitalCodeSystem);
            }
        }

        /// <summary>
        /// Gets a list containing prescription and dispense document classes.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="27 November 2013">Initial version.</change>
        /// </history>
        private List<string> PrescriptionAndDispenseDocumentClasses
        {
            get
            {
                return this.Settings.GetSettingValue(Setting.SettingCodes.PcehrViewPDDocumentClasses).Split(',').ToList();
            }
        }

        /// <summary>
        /// Gets a list containing document classes for which service dates should be displayed.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="29 November 2013">Initial version.</change>
        /// </history>
        private List<string> ServiceDateDocumentClasses
        {
            get
            {
                return this.Settings.GetSettingValue(Setting.SettingCodes.PcehrViewServiceDateDocumentClasses).Split(',').ToList();
            }
        }

        /// <summary>
        /// Gets an integer value representing the number of months the "From" date will be offset from the current date by default.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="17 December 2013">Initial version.</change>
        /// </history>
        private int FromDateOffsetMonths
        {
            get
            {
                return int.Parse(this.Settings.GetSettingValue(Setting.SettingCodes.PcehrViewFromDateOffsetMonths));
            }
        }

        #endregion

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="patientRepository">Patient repository to be used by this controller.</param>
        /// <param name="pcehrViewRepository">PCEHR view repository to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        public PcehrViewController(IHospitalRepository hospitalRepository, IPatientRepository patientRepository, IPcehrViewRepository pcehrViewRepository, ISettingsRepository settingsRepository)
        {
            this.HospitalRepository = hospitalRepository;
            this.PatientRepository = patientRepository;
            this.PcehrViewRepository = pcehrViewRepository;
            using (settingsRepository)
            {
                this.Settings = settingsRepository.GetSettings();
            }
        }

        #endregion

        #region Methods

        #region Actions

        #region Hospitals / Patients at Hospital

        /// <summary>
        /// Display a list of hospitals for selection.
        /// </summary>
        /// <returns>View</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 January 2014">Initial version.</change>
        /// </history>
        [HttpGet]
        public ActionResult Hospitals()
        {
            return Patients(null);
        }

        /// <summary>
        /// Display a list of patients for a selected hospital.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <returns>View</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        [HttpGet]
        public ActionResult Patients(string hospitalId)
        {
            //Create ViewModel:
            var m = new PatientsViewModel() { HospitalId = hospitalId };
            this.LoadCurrentContext(m);

            //Load reference data:
            var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.HospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

            if ((hospitals != null) && (hospitals.Count() > 0))
            {
                //Update ViewModel with reference data:
                m.Hospitals = hospitals.ToSelectListItems(h => h.Code, h => h.Name);
            }
            else
            {
                m.Messages.Add("No hospitals available for selection.", MessageLevel.Error);
            }

            //Don't allow load for all patients:
            if (hospitalId != null)
            {
                //Load patients for selected hospital.
                var response = this.PatientRepository.ListPatientsCurrentlyInHospital(this.DefaultHospitalCodeSystem, hospitalId, true, this.GetCurrentUserDetails());

                // Ensure loading was successful.
                if (response.IsSuccessful)
                {
                    if (response.Data.PatientInHospitalList.Count > 0)
                    {
                        //Update ViewModel with patients.
                        m.Patients.AddRange(ObjectMapper.Map<IEnumerable<PatientViewModel>>(response.Data.PatientInHospitalList));
                    }
                    else
                    {
                        m.Messages.Add("No PCEHR data is available for patients at the selected hospital.", MessageLevel.Information);
                    }
                }
                else
                {
                    string errorMessage = "Failed to retrieve patients for the selected hospital.";
                    //Log details:
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));
                    //Display error message.
                    this.SetAjaxErrorResponseCode();
                    m.Messages.Add(errorMessage, MessageLevel.Error);
                }
            }

            return View("Patients", m);
        }

        #endregion

        #region Patient Summary

        /// <summary>
        /// Display a summary for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <returns>View</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        [HttpGet]
        [PcehrAccess]
        public ActionResult PatientSummary(string hospitalId, string patientId)
        {
            //Create ViewModel:
            var m = new PatientSummaryViewModel() { HospitalId = hospitalId, PatientId = patientId };
            this.LoadCurrentContext(m);

            //Load documents for selected patient.
            var docs = this.PcehrViewRepository.ListActiveDocuments(
                new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem),
                this.GetCurrentUserDetails());

            //Copy over response messages.
            m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(docs.Messages));
            
            // Load document list
            if (docs.IsSuccessful)
            {
                if (docs.Data.DocumentList != null && docs.Data.DocumentList.Any())
                {
                    var includedDocuments = docs.Data.DocumentList.Where(d => !this.PrescriptionAndDispenseDocumentClasses.Contains(d.DocumentClassCode));
                    //Update ViewModel with documents & document categories.
                    m.Documents.AddRange(ObjectMapper.Map<IEnumerable<DocumentViewModel>>(includedDocuments));
                    m.DocumentCategories.AddRange(
                        includedDocuments.GroupBy(d => new { d.DocumentClassCode, d.DocumentClassName })
                            .Select(
                                cat => new DocumentCategoryViewModel()
                                {
                                    CategoryCode = cat.Key.DocumentClassCode,
                                    CategoryName = cat.Key.DocumentClassName,
                                    Count = cat.Count(),
                                    AreServiceDatesRelevant = this.ServiceDateDocumentClasses.Contains(cat.Key.DocumentClassCode)
                                }
                            ).OrderBy(cat => cat.CategoryName)
                        );
                    m.PrescriptionDispenseDocumentCount = docs.Data.DocumentList.Count(d => this.PrescriptionAndDispenseDocumentClasses.Contains(d.DocumentClassCode));
                }
                else
                {
                    m.Messages.Add("No documents are available for the selected patient.", MessageLevel.Information);
                }
            }
            else
            {
                string errorMessage = "Failed to retrieve documents for the selected patient.";
                //Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, docs.Messages.AsString())));
                //Display error message.
                this.SetAjaxErrorResponseCode();
            }

            return View(m);
        }

        #endregion

        #region Document / Content

        /// <summary>
        /// Display a specific document for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="repositoryId">Identifier of the document repository.</param>
        /// <param name="documentId">Identifier of the document in the repository.</param>
        /// <returns>
        /// View
        /// </returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        ///   </history>
        [HttpGet]
        [PcehrAccess]
        public ActionResult Document(string hospitalId, string patientId, string repositoryId, string documentId)
        {
            //Create ViewModel:
            PcehrViewModelBase m = new PcehrViewModelBase() { HospitalId = hospitalId, PatientId = patientId, RepositoryId = repositoryId, DocumentId = documentId };
            this.LoadCurrentContext(m);

            // If Ajax request return a PartialView
            if (Request.IsAjaxRequest())
            {
                return PartialView(m);
            }

            return View(m);
        }

        [HttpGet]
        public ActionResult DocumentContentView(string hospitalId, string patientId, string repositoryId, string documentId, string filename)
        {
            // Check if request is for a filename that the filename is valid 
            if (!String.IsNullOrWhiteSpace(filename))
            {
                // If filename is missing an extension, or has a directory, then it is invalid
                bool hasDirectory = !String.IsNullOrWhiteSpace(Path.GetDirectoryName(filename));
                if (!Path.HasExtension(filename) || hasDirectory)
                {
                    return HttpNotFound();
                }
            }

            // Retrieve document
            var doc = this.PcehrViewRepository.GetDocument(
                new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem),
                this.GetCurrentUserDetails(), 
                new DocumentRequest() { RepositoryUniqueId = repositoryId, DocumentUniqueId = documentId, SaveDocument = false }
                );

            // If unsuccessful lookup then return 404
            if (!doc.IsSuccessful)
            {
                return HttpNotFound();
            }

            // Looking for rendered HTML (no filename or filename is "CDA_ROOT.html")
            if (filename == null || filename.Equals("CDA_ROOT.html", StringComparison.OrdinalIgnoreCase))
            {
                return Content(doc.Data.TransformToHtml(Properties.Resources.NEHTA_Generic_CDA_Stylesheet_1_2_8));
            }
            
            // Looking for XML source
            if (filename.Equals("CDA_ROOT.xml", StringComparison.OrdinalIgnoreCase))
            {
                return File(doc.Data.Document, "text/xml");
            }

            // Looking for an attachment
            {
                var attachment = doc.Data.Attachments.FirstOrDefault(a => a.FileName.Equals(filename, StringComparison.OrdinalIgnoreCase));

                // If nothing found, return 404
                if (attachment == null)
                {
                    return HttpNotFound();
                }

                // Return attachment with filename and inferred MIME type, attempt to display inline rather than prompt download
                return this.FileInlineInferMime(attachment.Contents, filename);
            }
        }

        /// <summary>
        /// Display a specific document for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="repositoryId">Identifier of the document repository.</param>
        /// <param name="documentId">Identifier of the document in the repository.</param>
        /// <returns>View</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        [HttpGet]
        public ActionResult DocumentContent(string hospitalId, string patientId, string repositoryId, string documentId)
        {
            DocumentContentViewModel m = new DocumentContentViewModel() { HospitalId = hospitalId, PatientId = patientId, RepositoryId = repositoryId, DocumentId = documentId };
            this.LoadCurrentContext(m);

            // Retrieve document
            var doc = this.PcehrViewRepository.GetDocument(
                new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem),
                this.GetCurrentUserDetails(), 
                new DocumentRequest() { RepositoryUniqueId = repositoryId, DocumentUniqueId = documentId, SaveDocument = false }
                );

            //Copy over response messages.
            m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(doc.Messages));

            // Save and transform if a document was retrieved
            if (doc.IsSuccessful)
            {
                m.DocumentUrl = Url.Action("DocumentContentView", new { hospitalId, patientId, repositoryId, documentId, filename="CDA_ROOT.html" });
                m.DownloadTimestamp = doc.ReceiptTimestamp;
            }
            else
            {
                string errorMessage = "Failed to retrieve content for the selected document.";
                //Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, doc.Messages.AsString())));
                //Display error message.
                this.SetAjaxErrorResponseCode();
            }

            return PartialView(m);
        }

        #endregion

        #region Prescription Dispense View / Content

        /// <summary>
        /// Display a prescription and dispense view page for a selected patient.
        /// </summary>
        /// <param name="request">The request.</param>
        /// <returns>
        /// View
        /// </returns>
        [HttpGet]
        [PcehrAccess]
        public ActionResult PrescriptionDispenseView(PrescriptionDispenseViewModel request)
        {
            this.LoadCurrentContext(request);

            request.GroupByOptions = new Dictionary<string, string>() 
            { 
                { "prescription", "Prescription" },
                { "genericName", "Generic Name" },
                { "pbsItemCode", "PBS Item Code" },
                { "brandName", "Brand Name" },
            }.ToSelectListItems(i => i.Key, i => i.Value);

            if (!request.FromDate.HasValue || !request.ToDate.HasValue)
            {
                // Don't make the date fields required on page load of view
                ModelState.Clear();
            }
            
            request.FromDate = request.FromDate ?? DateTime.Today.AddMonths(this.FromDateOffsetMonths);
            request.ToDate = request.ToDate ?? DateTime.Today;
            request.GroupBy = request.GroupBy ?? request.GroupByOptions.First().Value;

            return PartialView(request);
        }

        /// <summary>
        /// Renders the content section of a prescription and dispense view page for a selected patient.
        /// </summary>
        /// <param name="request">The request.</param>       
        /// <returns>View</returns>
        public ActionResult PrescriptionDispenseViewContent(PrescriptionDispenseViewModel request)
        {
            // Create ViewModel (with breadcrumbs)
            DocumentContentViewModel m = new DocumentContentViewModel() { HospitalId = request.HospitalId, PatientId = request.PatientId };
            this.LoadCurrentContext(m);

            // If invalid, return with errors
            if (!ModelState.IsValid)
            {
                // Add ModelState errors to 'Messages'
                m.Messages.AddRange(ModelState.ToErrorDictionary().SelectMany(msg => msg.Value).Select(e => new ViewMessage(e, MessageLevel.Error)));
                this.SetAjaxErrorResponseCode();
                return PartialView("DocumentContent", m);
            }
            
            // Retrieve view
            var view = PcehrViewRepository.GetView(
                new CommonSchemas.PatientIdentifier.Mrn(request.PatientId, request.HospitalId, DefaultHospitalCodeSystem),
                this.GetCurrentUserDetails(),
                new PrescriptionAndDispenseViewRequest() { FromDate = request.FromDate.Value, ToDate = request.ToDate.Value }
            );

            //Copy over response messages.
            m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(view.Messages));
            
            // Return URL of view
            if (view.IsSuccessful)
            {
                m.DocumentUrl = Url.Action("PrescriptionDispenseViewContentView", new { FromDate = request.FromDate.Value.ToString("dd/MM/yyyy"), ToDate = request.ToDate.Value.ToString("dd/MM/yyyy"), GroupBy = request.GroupBy });
                m.DownloadTimestamp = view.ReceiptTimestamp;
            }
            else
            {
                string errorMessage = "Failed to retrieve content for the selected view.";
                //Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, view.Messages.AsString())));
                //Display error message.
                this.SetAjaxErrorResponseCode();
            }

            return PartialView("DocumentContent", m);
        }

        [HttpGet]
        public ActionResult PrescriptionDispenseViewContentView(PrescriptionDispenseViewModel request)
        {
            // If invalid, return not found
            if (!ModelState.IsValid)
            {
                return HttpNotFound();
            }

            // Retrieve view
            var view = PcehrViewRepository.GetView(
                new CommonSchemas.PatientIdentifier.Mrn(request.PatientId, request.HospitalId, DefaultHospitalCodeSystem),
                this.GetCurrentUserDetails(),
                new PrescriptionAndDispenseViewRequest() { FromDate = request.FromDate.Value, ToDate = request.ToDate.Value }
            );

            // If error retrieving view then return not found
            if (!view.IsSuccessful)
            {
                return HttpNotFound();
            }

            string renderedHtml = view.Data.TransformToHtml(
                    Properties.Resources.NEHTA_PCEHR_Prescription_and_Dispense_View_CDA_Stylesheet_1_1_0,
                    Url.Action("Document", new { request.HospitalId, request.PatientId, repositoryId = "Remove", documentId = "Remove" }).Replace("Remove/Remove", String.Empty), // TODO Better way to get root URL
                    request.PatientId,
                    request.FromDate.Value,
                    request.ToDate.Value,
                    request.GroupBy);

            // Return HTML
            return Content(renderedHtml);
        }

        #endregion

        #region Gain Access

        /// <summary>
        /// Gain access to the PCEHR for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="returnUrl">The original URL.</param>
        /// <returns>View</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
        /// </history>
        [HttpGet]
        public ActionResult GainAccess(string hospitalId, string patientId, string returnUrl)
        {
            //Create ViewModel:
            var m = new GainAccessViewModel() { HospitalId = hospitalId, PatientId = patientId, ReturnUrl = returnUrl };
            this.LoadCurrentContext(m);

            //Add message provided by PcehrAccessFilter / PcehrAccessManager if present.
            if (this.TempData.ContainsKey("PcehrAccessMessage") && (!string.IsNullOrEmpty((string)this.TempData["PcehrAccessMessage"])))
            {
                m.Messages.Add((string)this.TempData["PcehrAccessMessage"], MessageLevel.Warning);
            }

            // If Ajax request return a PartialView
            if (Request.IsAjaxRequest())
            {
                return PartialView(m);
            }

            return View(m);
        }

        /// <summary>
        /// Gain access to the PCEHR for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="returnUrl">The original URL.</param>
        /// <param name="accessType">Access type selected by the user.</param>
        /// <param name="accessCode">Access code (if specified).</param>
        /// <returns>View</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
        /// </history>
        [HttpPost]
        public ActionResult GainAccess(string hospitalId, string patientId, string returnUrl, AccessType accessType, string accessCode = null)
        {
            //Create ViewModel:
            var m = new GainAccessViewModel() { HospitalId = hospitalId, PatientId = patientId, ReturnUrl = returnUrl, AccessType = accessType, AccessCode = accessCode };
            this.LoadCurrentContext(m);

            //Validate:
            if (accessType == AccessType.WithCode)
            {
                if (string.IsNullOrEmpty(accessCode))
                {
                    this.ModelState.AddModelError("AccessCode", "A non-empty Access Code must be provided when attempting to gain access using an access code.");
                }
            }

            if (this.ModelState.IsValid)
            {
                //Attempt to gain access:

                HIPS.Web.Components.ServiceModel.ServiceResponse<GainPcehrAccessResponse> result = null;

                switch (accessType)
                {
                    case AccessType.WithCode:
                    {
                        result = this.PcehrViewRepository.GainAccessWithCode(new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem), accessCode, this.GetCurrentUserDetails());
                        break;
                    }
                    case AccessType.Emergency:
                    {
                        result = this.PcehrViewRepository.GainAccessEmergency(new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem), this.GetCurrentUserDetails());
                        break;
                    }
                    default:
                    {
                        break;
                    }
                }

                if (result != null)
                {
                    if (result.IsSuccessful)
                    {
                        return Redirect(returnUrl);
                    }
                    else
                    {
                        string errorMessage = "Failed to gain access for the selected patient.";
                        //Log details:
                        Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, result.Messages.AsString())));
                        //Copy over response messages.
                        m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(result.Messages));
                    }
                }

            }

            //Invalid outcome, redisplay view.
            return View(m);
        }

        #endregion

        #endregion

        #region Helpers

        /// <summary>
        /// Loads required context into the provided view model.
        /// </summary>
        /// <param name="model">View model to load context into.</param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
        /// </history>
        private void LoadCurrentContext(PcehrViewModelBase model)
        {

            //Load current hospital.
            if (! string.IsNullOrEmpty(model.HospitalId))
            {
                var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.HospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

                if ((hospitals != null) && (hospitals.Count() > 0))
                {
                    model.CurrentHospital = hospitals.FirstOrDefault(h => h.Code == model.HospitalId);;
                }
            }

            //Load current patient.
            if (!string.IsNullOrEmpty(model.PatientId))
            {
                model.CurrentPatient = ObjectMapper.Map<PatientViewModel>(this.PatientRepository.GetPatientInHospital(new CommonSchemas.PatientIdentifier.Mrn(model.PatientId, model.HospitalId, this.DefaultHospitalCodeSystem), this.GetCurrentUserDetails()));
            }

            //Load current document.
            if ((!string.IsNullOrEmpty(model.RepositoryId)) && (!string.IsNullOrEmpty(model.DocumentId)))
            {
                model.CurrentDocument = ObjectMapper.Map<DocumentViewModel>(this.PcehrViewRepository.GetDocumentMetaData(new CommonSchemas.PatientIdentifier.Mrn(model.PatientId, model.HospitalId, this.DefaultHospitalCodeSystem), model.RepositoryId, model.DocumentId, this.GetCurrentUserDetails()));
            }

        }

        #endregion

        #endregion

    }
}